/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_kern.c */


/*** as per OpenType 1.3 spec
 *
 *   'kern' Table
 *
 *   Should contain a single kerning pair subtable (format 0). Windows will
 *   not support format 2 (two-dimensional array of kern values by class); nor
 *   multiple tables (only the first format 0 table found will be used) nor
 *   coverage bits 0 through 4 (i.e. assumes horizontal data, kerning values,
 *   no cross stream, and override).
 *
 *   OpenType fonts containing CFF data do not support the 'kern' table and
 *   should therefore specify kerning data using the 'GPOS' table (LookupType=2).
 *
 ***/

#include "fs_itype.h"

#define SIZEOF_KST 6

/*********************************************************************/
/* set lfnt->kerning to be a pointer to the first format 0 kerning table
*  if (lfnt->memptr) it is a pointer to within the font
*  else it is the result of an FREAD
*/
FS_LONG load_kerning(_DS_ LFNT *lfnt)
{
    TTF *ttf;

    if (lfnt == 0)
        return STATE.error = ERR_NO_CURRENT_LFNT;

    ttf = (FS_VOID *)(lfnt->fnt);
    if (ttf == 0)
        return STATE.error = ERR_BAD_LFNT;

    /* do we need to load the first type 0 kerning sub-table ? */
    if ((KSTf0_data *)(lfnt->kerning) == 0 && ttf->kern_offset != 0)
    {
        if (lfnt->memptr)
        {
            FS_BYTE *p;
            FS_USHORT numTables, i;

            /* set a pointer to the memory resident sub-table */
            p = (FS_BYTE *)(lfnt->memptr + lfnt->data_offset + ttf->kern_offset);

            /* kerning table header */
            /*version = GET_xWORD(p);*/
            p += 2;
            numTables = GET_xWORD(p);
            p += 2;

            /* "p" now points to a kerning subtable header */
            for (i = 0; i < numTables; i++)
            {
                FS_USHORT version, length;

                /* kerning subtable header */
                version = GET_xWORD(p);
                p += 2;
                length = GET_xWORD(p);
                p += 2;
                /*coverage= GET_xWORD(p);*/
                p += 2;

                /* "p" now points to the kerning subtable data */
                if (version == 0)
                {
                    lfnt->kerning = p;
                    return SUCCESS;
                }
                p += (length - SIZEOF_KST); /* already traversed the header */
            }
        }
        else
        {
            FS_ULONG ulNumTables, i;
            FS_USHORT numTables;

            /* read the sub-table into RAM, set a pointer to it */
            FS_BYTE p4[4], *p;
            FS_ULONG offset = ttf->kern_offset;

            /* read KTH from disk */
            p = &p4[0];
            ttf_read_buf(_PS_ ttf, offset, 4, p);

            /*version = GET_xWORD(p);*/
            numTables = GET_xWORD(p + 2);

            /* next ttf_read to be after kerning table header; KST header */
            offset += 4;

            /* following is trick to fix a Coverity tainted variable complaint */
            ulNumTables = numTables;
            if (ulNumTables > 65535)
            {
                return STATE.error = ERR_BAD_GLYF_FORMAT;
            }

            /* read KST's from disk */

            for (i = 0; i < ulNumTables; i++)
            {
                FS_BYTE p6[SIZEOF_KST];
                FS_USHORT version, length;

                ttf_read_buf(_PS_ ttf, offset, SIZEOF_KST, &p6[0]);

                version = GET_xWORD(p6);
                length  = GET_xWORD(p6 + 2);
                /*coverage= GET_xWORD(p6+4);*/

                /* next ttf_read to be after KST header; KST data */
                offset += SIZEOF_KST;

                if (version == 0)
                {
                    /* read the version 0 subtable data into RAM */
#ifdef FS_MEM_DBG
                    STATE.memdbgid = "lfnt->kerning";
#endif
                    lfnt->kerning = ttf_read(_PS_ ttf, offset, length - SIZEOF_KST);

                    return SUCCESS;
                }
                /* next ttf_read to be after KST data; next KST header */
                offset += (length - SIZEOF_KST);
            }
        }
    }
    return SUCCESS;
}
/*********************************************************************/
FS_VOID delete_kern(_DS_ LFNT *lfnt)
{
    KSTf0_data *data = (KSTf0_data *)(lfnt->kerning);

    if (data && lfnt->memptr == 0)
    {
        FSS_free(_PS_ data);
        lfnt->kerning = 0;
    }
}

/*********************************************************************/
FS_LONG FSS_get_kerning(_DS_ FS_ULONG id1, FS_ULONG id2, FS_FIXED *_dx, FS_FIXED *_dy)
{
    SFNT *sfnt;
    LFNT *lfnt;
    TTF *ttf;
    FS_USHORT index1, index2;

    /* initial fixed point, transformed kerning pair */
    *_dx = *_dy = 0;

    /* change from character codes to glyph indices */
    index1 = map_char(_PS_ id1, 0);
    if (STATE.error)
        return STATE.error;
    sfnt = STATE.cur_sfnt;
    if (sfnt == 0)
        return STATE.error = ERR_NO_CURRENT_SFNT;

    /* gotta be a current sfnt, lfnt and ttf */
    lfnt = (LFNT *)(sfnt->lfnt);
    if (lfnt == 0)
        return STATE.error = ERR_NO_CURRENT_LFNT;

    if (lfnt->fnt_type == PFR_TYPE)
    {
        STATE.error = ERR_TABLE_UNSUPPORTED;
        return 0;
    }

    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        STATE.error = ERR_TABLE_UNSUPPORTED;
        return 0;
    }
    ttf = (FS_VOID *)(lfnt->fnt);

    if (ttf->kern_offset == 0)
        return SUCCESS;
    if (lfnt->kerning == 0)
        load_kerning(_PS_ lfnt);
    if (lfnt->kerning == 0)
        return SUCCESS;        /* not really ... there was no format 0 subtable */

    index2 = map_char(_PS_ id2, 0);
    if (STATE.error)
        return STATE.error;

    /* indices must be from the same component font */
    if (STATE.cur_sfnt != sfnt)
        return SUCCESS;  /* no kerning pair available */


    /* binary search the triples */
    {
        int num, lo, hi;
        FS_BYTE *bp, *ptr;

        bp = (FS_BYTE *)(KSTf0_data *)(lfnt->kerning);    /* the first format 0 *)(sub-table  */
        num = GET_xWORD(bp);            /* number of <left, right, value> triples */
        bp += 8;                        /* now points to start of triples         */
        lo = 0;
        hi = num - 1;

        while (lo <= hi)
        {
            FS_USHORT mid, first, second;

            mid = (FS_USHORT)(lo + hi) / 2;
            ptr = bp + (mid << 2) + (mid << 1);    /* byte pointer to the mid-th triple */
            first = GET_xWORD(ptr);
            ptr += 2;
            second = GET_xWORD(ptr);
            ptr += 2;

            if (first > index1 || (first == index1 && second > index2))
                hi = mid - 1;
            else if (first < index1 || (first == index1 && second < index2))
                lo = mid + 1;
            else
            {
                if (first == index1 && second == index2)
                {
                    FS_SHORT dx = (FS_SHORT) GET_xWORD(ptr);
                    if (STATE.flags & FLAGS_KERNING_SCALE_OFF)
                    {
                        *_dx = dx;
                        *_dy = dx;
                    }
                    else
                    {
                        int unitsPerEm = ttf->head->unitsPerEm;

                        *_dx = LongMulDiv(sfnt->user_scale[0], dx, unitsPerEm);
                        *_dy = LongMulDiv(sfnt->user_scale[2], dx, unitsPerEm);
                    }
                }
                break;
            }
        }
    }
    return SUCCESS;
}
/*********************************************************************/
